<?php

namespace App\Filament\Resources\Articles\Schemas;

use App\Models\ArticleGroup;
use App\Models\Setting;
use App\Rules\ArticlePriceValidation;
use Filament\Actions\Action;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;

class ArticleForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                TextInput::make('name')
                    ->hint('z.B. Dirndl lang mit rot, gold, grün karierter Schürze')
                    ->label('Artikel Name')
                    ->minLength(3)
                    ->maxLength(100)
                    ->required(),
                TextInput::make('price')
                    ->label('Preis')
                    ->required()
                    ->numeric()
                    ->step(0.50)
                    ->minValue(0.5)
                    ->maxValue(99999) // static max value (checked in js)
                    ->rules(fn (Get $get): array => [
                        new ArticlePriceValidation($get('article_group_id')), // dynamic max value (checked on server)
                    ])
                    ->prefix('€')
                    ->hintAction(
                        Action::make('price_help')
                            ->icon('heroicon-m-question-mark-circle')
                            ->label('Hilfe bei der Preisfindung')
                            ->modalHeading('Hilfe bei der Preisfindung')
                            ->modalContent(fn () => new \Illuminate\Support\HtmlString(
                                '<div class="fi-prose">'.str(Setting::getHelpTextPrice())->sanitizeHtml().'</div>'
                            ))
                            ->modalSubmitAction(false)
                            ->modalCancelAction(false)
                    ),
                Select::make('article_group_id')
                    ->label('Artikel Gruppe')
                    ->required()
                    ->searchable()
                    ->options(ArticleGroup::all()->pluck('name', 'id')),
                TextInput::make('size')
                    ->hint('Bitte XS/S/M bzw. 46/48/u.s.w verwenden')
                    ->maxLength(7)
                    ->label('Größe'),
                TextInput::make('brand')
                    ->maxLength(50)
                    ->label('Marke'),
                Select::make('lables_needed')
                    ->label('Etiketten Anzahl')
                    ->hintAction(
                        Action::make('price_help')
                            ->icon('heroicon-m-question-mark-circle')
                            ->label('Hilfe Etiketten')
                            ->modalHeading('Hilfe Etiketten')
                            ->modalContent(fn () => new \Illuminate\Support\HtmlString(
                                '<div class="fi-prose">'.str(Setting::getHelpTextLabelsNeeded())->sanitizeHtml().'</div>'
                            ))
                            ->modalSubmitAction(false)
                            ->modalCancelAction(false)
                    )
                    ->options([
                        1 => '1 Etikett',
                        2 => '2 Etiketten',
                        3 => '3 Etiketten',
                        4 => '4 Etiketten',
                        5 => '5 Etiketten',
                    ])
                    ->required()
                    ->default(1),
            ]);
    }
}
