<?php

namespace App\Models;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use OwenIt\Auditing\Contracts\Auditable;
use Picqer\Barcode\Renderers\PngRenderer;
use Picqer\Barcode\Types\TypeEan13;

class Article extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;

    protected $auditInclude = [
        'name', 'price', 'article_group_id', 'size', 'brand', 'lables_needed',
    ];

    // elements possible to set on update or create
    protected $fillable = [
        'name', 'price', 'article_group_id', 'size', 'brand', 'lables_needed',
    ];

    protected static function booted(): void
    {
        static::addGlobalScope('tmk_articles_only', function (Builder $builder) {
            // this model is filtered down to only articles with the id of the current user
            if (Auth::check()) {
                $builder->where('user_id', '=', Auth::id());
            } else {
                $builder->where('user_id', -5000); // invalid value will not return any data
            }
        });

        static::creating(function (Article $article) {

            // enforce max articles per customer limit
            $max_customer = Setting::getMaxArticlesPerCustomer();
            if (static::count() >= $max_customer) {
                throw ValidationException::withMessages([
                    'limit_customer' => "Die maximale Artikelanzahl pro Kunde wurde mit {$max_customer} Stück erreicht.",
                ]);
            }

            // enforce max articles total limit
            $max_total = Setting::getMaxArticlesTotal();
            if (DB::table('articles')->count() >= $max_total) {
                throw ValidationException::withMessages([
                    'limit_total' => "Die maximale Artikelanzahl in der Vorerfassung wurde mit {$max_total} Stück erreicht.",
                ]);
            }

            // automatically set some properties before creating an article
            $article->user_id = Auth::id();

        });

        static::created(function (Article $article) {
            // automatically generate ean after creating the article (we need to know the db id)
            $article->ean = self::calculateEan($article->id, $article->imported_year);
            $article->save();
        });

        // triggered before creating or updating
        static::saving(function (Article $article) {
            // enforce pre recording times
            if (Setting::getPreRecordingDisabled()) {
                throw ValidationException::withMessages([
                    'not_active' => 'Die Vorerfassung ist aktuell nicht möglich.',
                ]);
            }

            // if ean has not generated before try to regenerate it now
            if (empty($article->ean) && ! empty($article->id)) {
                $article->ean = self::calculateEan($article->id, $article->imported_year);
                $article->save();
            }

        });

        // triggered before delete
        static::deleting(function ($model) {
            // enforce pre recording times
            if (Setting::getPreRecordingDisabled()) {
                throw ValidationException::withMessages([
                    'not_active' => 'Die Vorerfassung ist aktuell nicht möglich.',
                ]);
            }
        });
    }

    /**
     * Calculate tauschmarkt ean from article id
     */
    public static function calculateEan(int $articleId, ?int $yearOverwrite = null)
    {
        if (empty($yearOverwrite)) {
            $year = date('Y');
        } elseif ($yearOverwrite >= 2025 && $yearOverwrite <= intval(date('Y'))) {
            $year = $yearOverwrite;
        } else {
            throw new Exception('calculateEan yearOverwrite '.$yearOverwrite.' is not valid');
        }

        $ean12 = $year.str_pad($articleId, 8, '0', STR_PAD_LEFT);
        $ean13 = $ean12.self::calculateEan13ChecksumDigit($ean12);

        return $ean13;
    }

    /**
     * Calculate ean13 checksum
     */
    public static function calculateEan13ChecksumDigit(string $digits): int
    {
        $digits = preg_replace('/\D/', '', $digits);
        $reversedDigits = str_split(strrev($digits));
        $sum = 0;
        foreach ($reversedDigits as $index => $digit) {
            $multiplier = ($index % 2 === 0) ? 3 : 1;
            $sum += $digit * $multiplier;
        }
        $checksum = (10 - ($sum % 10)) % 10;

        return $checksum;
    }

    public function articleGroup(): BelongsTo
    {
        return $this->belongsTo(ArticleGroup::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Converts EAN13 to an png barcode picture
     */
    public static function EAN13toPng(string $ean)
    {
        // see also https://github.com/picqer/php-barcode-generator

        $barcode = (new TypeEan13)->getBarcode($ean);
        $renderer = new PngRenderer;
        $barcodeData = $renderer->render($barcode, $barcode->getWidth() * 2, 20);
        $base64Barcode = base64_encode($barcodeData);

        return '<img class="barcode-img" src="data:image/png;base64,'.$base64Barcode.'" alt="barcode">';

    }

    /**
     * Return article lable collection for printing
     * some items are duplicated because of lable amount > 1
     */
    public static function getArticleLables()
    {
        $articles = Article::with(['articleGroup', 'user'])->where('user_id', '=', Auth::id())->get();

        $labels = $articles->flatMap(function ($item) {

            if ($item->lables_needed == 1) {
                $item->tmp_label_counter = 1;

                return [$item];
            } elseif ($item->lables_needed > 1) {

                $item->tmp_label_counter = 1;

                $replicates = [];
                for ($counter = 2; $counter <= $item->lables_needed; $counter++) {
                    $tmp = $item->replicate();
                    $tmp->id = $item->id;
                    $tmp->tmp_label_counter = $counter;
                    $replicates[] = $tmp;
                }

                return array_merge([$item], $replicates);
            }
        });

        return $labels;
    }
}
