<?php

namespace App\Models;

use DateTimeInterface;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Arr;

/**
 * @property string $auditable_type
 * @property ?int $user_id
 * @property mixed $auditable
 */
class Audit extends Model implements \OwenIt\Auditing\Contracts\Audit
{
    use \OwenIt\Auditing\Audit;

    /**
     * {@inheritdoc}
     */
    protected $guarded = [];

    /**
     * {@inheritdoc}
     */
    protected $casts = [
        'old_values' => 'json',
        'new_values' => 'json',
        // Note: Please do not add 'auditable_id' in here, as it will break non-integer PK models
    ];

    public function getSerializedDate(DateTimeInterface $date): string
    {
        return $this->serializeDate($date);
    }

    /**
     * get user obj for audit data
     *
     * @return BelongsTo<User, $this>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id', 'id');
    }

    public static function getSelectEvent(): array
    {
        return [
            'updated' => 'updated',
            'created' => 'created',
            'deleted' => 'deleted',
        ];
    }

    public static function getSelectAuditTypes(): array
    {
        return [
            'App\\Models\\Article' => 'Article',
            'App\\Models\\TmkArticle' => 'TmkArticle',
            'App\\Models\\Setting' => 'Setting',
            'App\\Models\\Post' => 'Post',
            'App\\Models\\ArticleGroup' => 'ArticleGroup',
            'App\\Models\\File' => 'File',
        ];
    }

    /**
     * Computed attribute value_comparison
     */
    protected function valueComparison(): Attribute
    {
        return Attribute::make(
            get: function (mixed $value, array $attributes) {
                $data_old = json_decode($attributes['old_values'] ?? '[]', true);
                $data_new = json_decode($attributes['new_values'] ?? '[]', true);

                $keys = array_merge(array_keys($data_old), array_keys($data_new));
                $keys = array_unique($keys);
                $return = [];
                foreach ($keys as $key) {
                    $return[] = 'KEY: '.$key.' OLD: '.Arr::get($data_old, $key).' NEW: '.Arr::get($data_new, $key).'';
                }

                return $return;
            },
        );
    }
}
