<?php

namespace App\Models;

use Database\Factories\PostFactory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * @property string $uuid
 * @property int $user_id
 * @property string $title
 * @property Collection<int, TranslationCache> $translations
 * @property Collection<int, Media> $pictures
 *
 * @method static PostFactory factory($count = null, $state = [])
 */
class Post extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;

    // elements possible to set on update or create
    protected $fillable = [
        'title', 'slug', 'content',
    ];

    protected $auditInclude = [
        'title',
        'content',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'content' => 'array',
        ];
    }

    public static string $cache_prefix = 'postcache';

    public static int $cache_fresh = 30 * 60; // 30min

    public static int $cache_stale = 24 * 60 * 60; // 24h

    public static function getCachedPostBySlug(string $slug): ?Post
    {
        $post = Cache::flexible(self::$cache_prefix.'_slug_'.$slug, [self::$cache_fresh, self::$cache_stale], function () use ($slug) {
            $post = DB::table('posts')->where('slug', $slug)->first('id');
            if ($post == null) {
                return null;
            } else {
                return $post;
            }
        });

        return $post;
    }

    /**
     * Event hook registration, see also https://laravel.com/docs/8.x/eloquent#events
     *
     * @return void
     */
    protected static function booted()
    {
        // on delete clear cache
        static::deleted(function ($post) {
            self::destroyCache($post->id, $post->slug);
        });
        // on create or update clear cache
        static::saved(function ($post) {
            self::destroyCache($post->id, $post->slug);
        });
    }

    public static function destroyCache(?string $slug_id = null): void
    {
        // forget cache on a specific post slug
        if ($slug_id != null) {
            Cache::forget(self::$cache_prefix.'_slug_'.$slug_id);
        }

        // forget cache on all post slugs
        if ($slug_id == null) {
            $posts = Post::all();
            foreach ($posts as $post) {
                Cache::forget(self::$cache_prefix.'_slug_'.$slug_id);
            }
        }
    }
}
