<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use OwenIt\Auditing\Contracts\Auditable;

class Setting extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;

    protected $auditInclude = [
        'value',
    ];

    // elements possible to set on update or create
    protected $fillable = [
        'type', 'value', 'key', 'description',
    ];

    public static function selectTypes()
    {
        return [
            'string' => 'string',
            'datetime' => 'datetime',
            'number' => 'number',
        ];
    }

    public static function getPreRecordingEnabled(): bool
    {
        if (Carbon::now('Europe/Vienna')->between(self::getPreRecordingStartTs(), self::getPreRecordingEndTs())) {
            return true;
        } else {
            return false;
        }
    }

    public static function getPreRecordingDisabled(): bool
    {
        return ! self::getPreRecordingEnabled();
    }

    public static function getSettingDynamic(string $key)
    {
        return Setting::where('key', $key)->value('value');
    }

    public static function getHelpTextPrice()
    {
        return self::getSettingDynamic('help_text_price');
    }

    public static function getHelpTextLabelsNeeded()
    {
        return self::getSettingDynamic('help_text_labels_needed');
    }

    public static function getPreRecordingStartTs()
    {
        return Carbon::createFromFormat('d.m.Y H:i:s', self::getSettingDynamic('pre_recording_start_ts'), 'Europe/Vienna');
    }

    public static function getPreRecordingEndTs()
    {
        return Carbon::createFromFormat('d.m.Y H:i:s', self::getSettingDynamic('pre_recording_end_ts'), 'Europe/Vienna');
    }

    public static function getMaxArticlesPerCustomer()
    {
        return (int) self::getSettingDynamic('max_articles_per_customer');
    }

    public static function getMaxArticlesTotal()
    {
        return (int) self::getSettingDynamic('max_articles_total');
    }
}
