<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use OwenIt\Auditing\Contracts\Auditable;

class TmkArticle extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;

    protected $auditInclude = [
        'name', 'price', 'article_group_id', 'size', 'brand', 'lables_needed',
    ];

    protected $table = 'articles'; // we will reuse the same table as the article model here

    // elements possible to set on update or create
    protected $fillable = [
        'name', 'price', 'article_group_id', 'size', 'brand', 'lables_needed',
    ];

    protected static function booted(): void
    {

        static::addGlobalScope('user_articles_only', function (Builder $builder) {
            // this model is filtered down to only articles with the id of tmk
            $tmk_user_id = config('app.user_id_tmk_koppl');
            $builder->where('user_id', '=', $tmk_user_id);
        });

        static::creating(function (TmkArticle $tmk_article) {
            // enforce max articles total limit
            $max_total = Setting::getMaxArticlesTotal();
            if (DB::table('articles')->count() >= $max_total) {
                throw ValidationException::withMessages([
                    'limit_total' => "Die maximale Artikelanzahl in der Vorerfassung wurde mit {$max_total} Stück erreicht.",
                ]);
            }

            // automatically set some properties before creating an article
            $tmk_user_id = config('app.user_id_tmk_koppl');
            $tmk_article->user_id = $tmk_user_id;

        });

        static::created(function (TmkArticle $tmk_article) {
            // automatically generate ean after creating the article (we need to know the db id)
            $tmk_article->ean = Article::calculateEan($tmk_article->id, $tmk_article->imported_year);
            $tmk_article->save();
        });

        // triggered before creating or updating
        static::saving(function ($model) {
            // enforce pre recording times
            if (Setting::getPreRecordingDisabled()) {
                throw ValidationException::withMessages([
                    'not_active' => 'Die Vorerfassung ist aktuell nicht möglich.',
                ]);
            }
        });

        // triggered before delete
        static::deleting(function ($model) {
            // enforce pre recording times
            if (Setting::getPreRecordingDisabled()) {
                throw ValidationException::withMessages([
                    'not_active' => 'Die Vorerfassung ist aktuell nicht möglich.',
                ]);
            }
        });
    }

    public function articleGroup(): BelongsTo
    {
        return $this->belongsTo(ArticleGroup::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
}
