<?php

/**
 * This is the model class for table "artikel".
 *
 * The followings are the available columns in table 'artikel':
 * @property string $ArtikelNr
 * @property string $Name
 * @property double $Preis
 * @property string $Kommentar
 * @property string $RefKundenNr
 * @property string $VerkauftFlag
 * @property string $VerkauftTS
 * @property string $ErfassungTS
 * @property string $Image
 * @property string $HappyHourFlag
 * @property string $ImportID
 */
class Artikel extends CActiveRecord
{
    /**
     * @return string the associated database table name
     */
    public function tableName()
    {
        return 'artikel';
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
            array('ArtikelNr, Name, Preis, RefKundenNr, HappyHourFlag', 'required'),
            array('ArtikelNr', 'unique', 'message' => 'Die Artikelnummer muss ohne führende Nullen eingegeben werden und darf nur einmal verwendet werden'), //stellt sicher dass die artikelnummer nicht nur in der db sondern auch im model all unique behandelt wird
            array('ArtikelNr', 'numerical'),
            array('ImportID', 'unique'), //importid muss wenn sie gesetzt wird eindeutig sein
            //array('Preis', 'numerical', 'integerOnly'=>FALSE,'numberPattern'=>'/^\s*[-+]?[0-9]*\,?[0-9]+([eE][-+]?[0-9]+)?\s*$/','message'=>'Preiseingabe ungültig'),
            //array('Preis', 'numerical', 'integerOnly'=>FALSE,'numberPattern'=>'/[1-9](?:\d{0,2})(?:,\d{3})*(?:\.\d*[1-9])?|0?\.\d*[1-9]|0/','message'=>'Preiseingabe ungültig v2'),
            //array('Preis', 'numerical', 'integerOnly'=>FALSE,'numberPattern'=>'/^(([1-9]*)|(([1-9]*).([0-9]*)))$/','message'=>'Preiseingabe ungültig v2'),
            array('Preis', 'validatePreis', 'message' => 'Preiseingabe ungültig, Format 12345,12 verwenden, Max=99999 Min=0,01'),

            //upload image
            array('Image', 'file', 'types' => 'jpg', 'allowEmpty' => true, 'on' => 'update', 'maxSize' => 16097152, 'tooLarge' => 'Das Bild darf maximal 15MB haben'), // this will allow empty field when page is update (remember here i create scenario update)

            array('ArtikelNr, RefKundenNr, RefWarengruppeID', 'length', 'max' => 20),
            array('Name, Image', 'length', 'max' => 100),
            array('Kommentar', 'length', 'max' => 300),
            array('VerkauftFlag, HappyHourFlag', 'length', 'max' => 5),
            array('HappyHourFlag, VerkauftFlag', 'in', 'range' => array('Ja', 'Nein'), 'allowEmpty' => false, 'message' => 'Es sind nur die Werte Ja oder Nein erlaubt'),
            array('Groesse', 'length', 'max' => 10),
            array('VerkauftRabattProz', 'numerical', 'integerOnly' => true, 'min' => 0, 'max' => 99, 'message' => 'Es sind nur Werte zwischen 0 und 99 gültig'),
            array('Groesse', 'validateGroesse', 'message' => 'Länge/Größe ungültig, Format XXS-XXL, Nummer 0-999,9 oder 42/44'),
            array('Marke', 'length', 'max' => 50),
            // The following rule is used by search().
            // todo Please remove those attributes that should not be searched.
            array('ArtikelNr, Name, Preis, Kommentar, RefKundenNr, VerkauftFlag, VerkauftTS, ErfassungTS, Image, Groesse, RefWarengruppeID, Marke, HappyHourFlag, ImportID', 'safe', 'on' => 'search'),

        );

        /*
        return array(
            array('Bezeichnung, Preis, RefKundenID', 'required'),
            array('Bezeichnung', 'length', 'max'=>255),
            //array('Preis', 'numerical'),
            // The following rule is used by search().
            // @todo Please remove those attributes that should not be searched.
            array('ID, Bezeichnung,Preis, RefKundenID', 'safe', 'on'=>'search'),
            
            //sicherstellen, dass nur nummerische werte eingegeben werden dürfen
            //array('Preis', 'match', 'numberPattern'=>'/^[\w\s,]+$/', 'message'=>'Tags can only contain word characters.'),
            //array('Preis', 'numerical', 'numberPattern'=>'/^\d+\.?\d*$/','message'=>'Preis muss als Zahl mit Beistrich eingegeben werden'),
            array('Preis','numerical',
                'integerOnly'=>false,
                'min'=>'0.01',
                'max'=>'99999',
                'tooSmall'=>'Preis beträgt mindestens 0,01€',
                'tooBig'=>'Preis beträgt maximal 99999€',
                'message'=>'Der Preis muss als Zahl mit Punkt angegeben werden'),  
                
        );*/
    }

    /**
     * Eigene Validierungsroutine um Preise zu validieren gültig Format 12345,12 oder 12345.11 verwenden, Max=99999 Min=0,01
     * 
     * @param mixed $attribute
     * @param mixed $params
     */
    public function validatePreis($attribute, $params)
    {
        //info von http://stackoverflow.com/questions/15516172/how-to-add-rules-using-yii-framework-in-model

        $error = null;
        $attribute_content = $this[$attribute];

        //ersetze komma durch punkt für validierung
        $attribute_content = str_replace(",", ".", $attribute_content);

        //daten eine zahl
        if (!is_numeric($attribute_content))
            $error = 1;

        //zahlk zwischen 0,01 und 99999
        if ((float)$attribute_content < 0.01 || $attribute_content > 99999)
            $error = 1;

        //maximal 5 corkomma und 2 nachkommastellen    
        $split = explode(".", $attribute_content);
        if ((isset($split[0]) && strlen($split[0]) > 5) || (isset($split[1]) && strlen($split[1]) > 2))
            $error = 1;

        if ($error) {
            //$this->addError('duration','validation message');
            $this->addError($attribute, $params['message']);
        }
    }

    /**
     * Eigene Validierungsroutine um Länge/Größe zu validieren, Format XXS-XXL oder Nummer 0-999,9
     * 
     * @param mixed $attribute
     * @param mixed $params
     */
    public function validateGroesse($attribute, $params)
    {
        //info von http://stackoverflow.com/questions/15516172/how-to-add-rules-using-yii-framework-in-model

        $valid_array = array();
        array_push($valid_array, "XXS");
        array_push($valid_array, "XS");
        array_push($valid_array, "S");
        array_push($valid_array, "M");
        array_push($valid_array, "L");
        array_push($valid_array, "XL");
        array_push($valid_array, "XXL");

        $error = null;
        $attribute_content = $this[$attribute];

        //ersetze komma durch punkt für validierung
        $attribute_content = str_replace(",", ".", $attribute_content);

        //daten eine zahl
        if (is_numeric($attribute_content)) {
            //maximal 3 vorkomma und 1 nachkommastelle    
            $split = explode(".", $attribute_content);
            if ((isset($split[0]) && strlen($split[0]) > 3) || (isset($split[1]) && strlen($split[1]) > 1))
                $error = 1;

            //zahlk zwischen 0 und 999.9
            if ((float)$attribute_content < 0 || $attribute_content > 999.9)
                $error = 1;
        } else if (is_numeric(str_replace("/", "", $attribute_content)))   //erlaube größeneingaben wie 42/44
        {
            $split = explode("/", $attribute_content);
            //erlaube nur dreistellige ganzzahlige zahlen
            if ((isset($split[0]) && strlen($split[0]) > 3) || (isset($split[1]) && strlen($split[1]) > 3))
                $error = 1;
        } else if ($attribute_content != "") //daten sind ein string
        {
            //daten sind immer uppercase
            $attribute_content = strtoupper($attribute_content);
            //wenn daten nicht in array -> error
            if (!in_array($attribute_content, $valid_array))
                $error = 1;
        } else if ($attribute_content == "") //daten sind leer
        {
            //alles ok da optionaler wert
        }


        if ($error) {
            //$this->addError('duration','validation message');
            $this->addError($attribute, $params['message']);
        }
    }


    /**
     * put your comment there...
     * 
     * @param mixed $id
     */
    public function writeChangeLog($id)
    {
        //lade daten aus db
        $artikel_alt = Artikel::model()->findByPk($id);

        //init var
        $changed = 0;
        $veraendert_wurden = "";

        //wenn neuer artikel
        if ($artikel_alt == null) {
            $veraendert_wurden .= "Neuer Artikel: ";

            //bei neuen artikeln alle elemente in log welche da sind anfügen
            foreach ($_POST['Artikel'] as $key => $value) {
                if ($value != "")
                    $veraendert_wurden .= "(" . $key . ") -> " . $value . ", ";
            }


            $changed = 1;
        } else  //bestehender artikel
        {
            //lade daten aus post und vergleiche daten
            if (isset($_POST['Artikel'])) {
                $artikel_neu = $_POST['Artikel'];

                foreach ($artikel_neu as $key => $val) {
                    $value_alt = $artikel_alt->$key;
                    if ($val != $value_alt) {
                        $changed = 1;
                        $veraendert_wurden .= $key . "(" . $value_alt . " -> " . $val . ")" . ", ";
                    }
                }
            } else {
                $changed = 0;
            }
        }

        //wenn unterschied reichere um IP/TS information an und speichere in Artikel_Aenderungslog
        if ($changed) {
            $model_log = new ArtikelAenderungslog;

            //initialisiere daten aus altem object wenn vorhanden
            if ($artikel_alt != null) {
                $model_log->attributes = $artikel_alt->attributes;
            } else //setze db defaultwerte damit log mit db werten 1:1 übereinstimmt
            {
                $model_log->VerkauftFlag = "Nein";
                $model_log->VerkauftRabattProz = "0";
                $model_log->ImportID = "0";
            }

            //überschreibe veränderte werte
            $model_log->attributes = $_POST['Artikel'];
            //setze spezialwerte
            $model_log->ChangeFields = substr($veraendert_wurden, 0, -2);
            $model_log->ChangeTS = date("Y-m-d H:i:s");
            $model_log->ChangeIP = Verkauf::model()->getCurrentClientIP();
            //speichern    
            $model_log->save();
        }
    }

    /**
     * Artikel Änderungen nur ArtikelNr und Infotext
     * 
     * @param mixed $ArtikelNr
     * @param mixed $text
     */
    public function writeChangeLogText($ArtikelNr, $text)
    {
        $model_log = new ArtikelAenderungslog;

        //setze spezialwerte
        $model_log->ChangeFields = $text;
        $model_log->ArtikelNr = $ArtikelNr;
        $model_log->ImportID = "0"; //mandatory  
        $model_log->RefKundenNr = "0"; //mandatory
        $model_log->ChangeTS = date("Y-m-d H:i:s");
        $model_log->ChangeIP = Verkauf::model()->getCurrentClientIP();

        //speichern    
        $model_log->save();
    }



    /**
     * @return array relational rules.
     */
    public function relations()
    {
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        //Infos von http://www.yiiframework.com/doc/guide/1.1/en/database.arr#disambiguating-column-names
        return array(
            'RefKundenID' => array(self::HAS_ONE, 'Kunden', 'ID')
        );
    }

    /**
     * @return array customized attribute labels (name=>label)
     */
    public function attributeLabels()
    {
        return array(
            'ArtikelNr' => 'Artikel Nr',
            'Name' => 'Name',
            'Preis' => 'Preis',
            'Kommentar' => 'Kommentar',
            'RefKundenNr' => 'KundenNr',
            'VerkauftFlag' => 'Verkauft',
            'VerkauftTS' => 'Verkauft am/um',
            'ErfassungTS' => 'Erfasst am/um',
            'Image' => 'Artikelbild',
            'Groesse' => 'Länge/Größe',
            'RefWarengruppeID' => 'Warengruppe',
            'Marke' => 'Marke',
            'HappyHourFlag' => 'HappyHour erlaubt',
            'VerkauftRabattProz' => 'Verkauf Reduziert (in %)',
            'ImportID' => 'Import ID',
        );
    }

    /**
     * Retrieves a list of models based on the current search/filter conditions.
     *
     * Typical usecase:
     * - Initialize the model fields with values from filter form.
     * - Execute this method to get CActiveDataProvider instance which will filter
     * models according to data in model fields.
     * - Pass data provider to CGridView, CListView or any similar widget.
     *
     * @return CActiveDataProvider the data provider that can return the models
     * based on the search/filter conditions.
     */
    public function search()
    {
        // todo Please modify the following code to remove attributes that should not be searched.

        $criteria = new CDbCriteria;

        $criteria->compare('ArtikelNr', $this->ArtikelNr, false);  //wird nur exakt zurück geben
        $criteria->compare('Name', $this->Name, true);
        $criteria->compare('Preis', $this->Preis, false); //nur exakt
        $criteria->compare('Kommentar', $this->Kommentar, true);
        $criteria->compare('RefKundenNr', $this->RefKundenNr, false); //wird nur exakt zurück geben
        $criteria->compare('VerkauftFlag', $this->VerkauftFlag, true);
        $criteria->compare('VerkauftTS', $this->VerkauftTS, true);
        $criteria->compare('ErfassungTS', $this->ErfassungTS, true);
        //sonderfall wenn image x gesucht wird, sonnst standardbehandlung
        if (isset($this->Image) && $this->Image == "x")
            $criteria->addCondition('Image IS NOT NULL');
        else
            $criteria->compare('Image', $this->Image, true);

        $criteria->compare('Groesse', $this->Groesse, true);
        $criteria->compare('RefWarengruppeID', $this->RefWarengruppeID, true);
        $criteria->compare('Marke', $this->Marke, true);
        $criteria->compare('HappyHourFlag', $this->HappyHourFlag, true);
        $criteria->compare('VerkauftRabattProz', $this->VerkauftRabattProz, false); //nur exakt
        $criteria->compare('ImportID', $this->ImportID, false);    //nur exakt

        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
            'pagination' => array('pageSize' => 50,), //50 artikel pro seite
            'sort' => array(
                'defaultOrder' => 'ErfassungTS DESC'
            )
        ));
    }

    /**
     * Liste aller Artikel auf einer seite sortiert nach KundenNr, Einschränkung nur auf VerkaufFlag
     */
    public function searchForPrintListeNichtVerkauft()
    {
        // todo Please modify the following code to remove attributes that should not be searched.

        $criteria = new CDbCriteria;
        $criteria->compare('VerkauftFlag', $this->VerkauftFlag, false); //teilweise übereinstimmung nein = 3. parameter auf false
        $criteria->compare('RefKundenNr', $this->RefKundenNr, false); //wird nur exakt zurück geben

        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
            'pagination' => array('pageSize' => 999999,), //99999 artikel pro seite
            'sort' => array(
                'defaultOrder' => 'RefKundenNr,RefWarengruppeID,Marke,Groesse ASC'
            )
        ));
    }

    /**
     * Liste von 100Artikeln pro seite Sortiert nach Artikelnr, Einschränkung nur auf Kunde
     * 
     */
    public function searchForPrint()
    {
        // todo Please modify the following code to remove attributes that should not be searched.

        $criteria = new CDbCriteria;
        $criteria->compare('RefKundenNr', $this->RefKundenNr, false); //teilweise übereinstimmung nein = 3. parameter auf false

        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
            'pagination' => array('pageSize' => 1000,), //1000 artikel pro seite
            'sort' => array(
                'defaultOrder' => 'ArtikelNr ASC'
            )
        ));
    }

    /**
     * Liste Artikel für Etikettendruck
     */
    public function searchForEtikettendruck()
    {
        $criteria = new CDbCriteria;
        $_POST = Helper::cleanAllInputData($_POST);

        //woraround falls direkt auf kunde aufgerufen damit nur daten dieses kunden angezeigt werden
        if (isset($_GET["id"]) && intval($_GET["id"]) > 0 && !isset($_POST["KundeVon"]))
            $_POST["KundeVon"] = intval($_GET["id"]);

        //prüfe auf Kunde exakt und von bis
        if (isset($_POST["KundeVon"]) && intval($_POST["KundeVon"]) != 0) {
            $criteria->addCondition('RefKundenNr >= ' . intval($_POST["KundeVon"]));
        }
        if (isset($_POST["KundeBis"]) && intval($_POST["KundeBis"]) != 0) {
            $criteria->addCondition('RefKundenNr <= ' . intval($_POST["KundeBis"]));
        }
        if (isset($_POST["KundeExakt"]) && intval($_POST["KundeExakt"]) != 0) {
            $criteria->compare('RefKundenNr', intval($_POST["KundeExakt"]));
        }
        //prüfe auf ArtikelNr von bis
        if (isset($_POST["ArtikelVon"]) && intval($_POST["ArtikelVon"]) != 0) {
            $criteria->addCondition('ArtikelNr >= ' . intval($_POST["ArtikelVon"]));
        }
        if (isset($_POST["ArtikelBis"]) && intval($_POST["ArtikelBis"]) != 0) {
            $criteria->addCondition('ArtikelNr <= ' . intval($_POST["ArtikelBis"]));
        }
        //prüfe auf happyhourflag ja unmd nein
        if (isset($_POST["HappyHour"]) && ($_POST["HappyHour"] == "Ja" || $_POST["HappyHour"] == "Nein")) {
            $criteria->compare('HappyHourFlag', $_POST["HappyHour"]);
        }
        //prüfe auf erfassungszeit
        if (isset($_POST["stdClass"]["ErfassungTSVon"]) && $_POST["stdClass"]["ErfassungTSVon"] != "") {
            $criteria->addCondition('ErfassungTS >= \'' . $_POST["stdClass"]["ErfassungTSVon"] . '\'');
        }
        if (isset($_POST["stdClass"]["ErfassungTSBis"]) && $_POST["stdClass"]["ErfassungTSBis"] != "") {
            $criteria->addCondition('ErfassungTS <= \'' . $_POST["stdClass"]["ErfassungTSBis"] . '\'');
        }


        return new CActiveDataProvider($this, array(
            'criteria' => $criteria,
            'pagination' => array('pageSize' => 100,), //100 artikel pro seite
            'sort' => array(
                'defaultOrder' => 'ArtikelNr ASC'
            )
        ));
    }

    /**
     * Berechnet die benötigten Werte für die Endabrechnung
     * 
     * @param mixed $KundenNr
     * @param mixed $add_verwaltungsaufwand wenn auf true wird verwaltungsaufwand mit berechnet, sonnst nicht
     * @return mixed
     */
    public function calcKunde($KundenNr, $add_verwaltungsaufwand)
    {
        //wieviel prozent der verkauften artikel werden als provision berechnet, aktuell 15
        //$calc_provision=15; 
        $calc_provision = (int)Parameter::model()->findByPk("abrechnung_provision_prozent")->Wert;
        //wieviel verwaltungsaufwand in € wird pro Artikel berechnet
        //$calc_verwaltungsaufwand=0.5;
        $calc_verwaltungsaufwand = (float)Parameter::model()->findByPk("abrechnung_betrag_verwaltungsaufwand")->Wert;

        //initialisieren des returns
        $return_array["sum"]["raw"] = 0;
        $return_array["sum"]["readable"] = "0,00 €";
        $return_array["sum"]["text"] = "Summe Verkauf";
        $return_array["provision"]["raw"] = 0;
        $return_array["provision"]["readable"] = "0,00 €";
        $return_array["provision"]["text"] = "einbehaltene Provision";
        $return_array["anzahl"]["raw"] = 0;
        $return_array["anzahl"]["readable"] = "0";
        $return_array["anzahl"]["Text"] = "Anzahl der Artikel";
        $return_array["verwaltung"]["raw"] = 0;
        $return_array["verwaltung"]["readable"] = "0,00 €";
        $return_array["verwaltung"]["text"] = "Verwaltungsaufwand";
        $return_array["total"]["raw"] = 0;
        $return_array["total"]["readable"] = "0,00 €";
        $return_array["total"]["text"] = "Gesamt Summe";
        $return_array["rundung"]["raw"] = 0;
        $return_array["rundung"]["readable"] = "0,00 €";
        $return_array["rundung"]["text"] = "Aufrundung auf 10Cent";

        //daten aus db abfragen
        $sql_anzahl = 0;
        $sql_anzahl = (int)Yii::app()->db->createCommand("SELECT count(*) as Anzahl  FROM artikel where RefKundenNr=" . $KundenNr . "")->queryScalar();

        $sql_summe = 0;
        //$sql_summe = (float)Yii::app()->db->createCommand("SELECT sum(preis) as Summe FROM artikel where RefKundenNr=".$KundenNr." and VerkauftFlag='Ja'")->queryScalar();
        //berücksichtige auch happy hour prozente hier
        $sql_summe = (float)Yii::app()->db->createCommand("SELECT sum(preis-(preis*VerkauftRabattProz/100)) as Summe FROM artikel where RefKundenNr=" . $KundenNr . " and VerkauftFlag='Ja'")->queryScalar();

        //werte berechnen
        if ($sql_anzahl != 0 && $add_verwaltungsaufwand == true) {
            $return_array["anzahl"]["raw"] = $sql_anzahl;
            $return_array["anzahl"]["readable"] = "" . $sql_anzahl . "";
            $return_array["verwaltung"]["raw"] = ($sql_anzahl * $calc_verwaltungsaufwand);
            $return_array["verwaltung"]["readable"] = number_format(($sql_anzahl * $calc_verwaltungsaufwand), 2, ",", "") . " €";
        }
        //werte berechnen
        if ($sql_summe != 0) {
            $return_array["sum"]["raw"] = $sql_summe;
            $return_array["sum"]["readable"] = number_format($sql_summe, 2, ",", "") . " €";
            $return_array["provision"]["raw"] = (float)(($sql_summe / 100) * $calc_provision);
            $return_array["provision"]["readable"] = number_format(((($sql_summe / 100) * $calc_provision)), 2, ",", "") . " €";
        }

        //gesamtsumme berechnen
        if ($sql_summe != 0 || $sql_anzahl != 0) {
            //rundung hinzufügen, es wird auf ganze 10cent aufgerundet
            $temp_total_raw = $return_array["sum"]["raw"] - $return_array["provision"]["raw"] - $return_array["verwaltung"]["raw"];
            $temp_total_raw_round = ceil($temp_total_raw * 10) / 10;
            $rundung_diff = $temp_total_raw_round - $temp_total_raw;
            $return_array["rundung"]["raw"] = $rundung_diff;
            $return_array["rundung"]["readable"] = number_format($rundung_diff, 2, ",", "") . " €";

            //total inklusive rundung
            $return_array["total"]["raw"] = $return_array["sum"]["raw"] - $return_array["provision"]["raw"] - $return_array["verwaltung"]["raw"] + $return_array["rundung"]["raw"];
            $return_array["total"]["readable"] = number_format($return_array["total"]["raw"], 2, ",", "") . " €";
        }

        return $return_array;
    }

    /**
     * liefert die nächste automatisch vom system vergebene artikelnummer zurück - für artikel import
     */
    public function getAutoArtikelNr()
    {
        //parameter einlesen
        $artnr_start = (float)Parameter::model()->findByPk("artikel_autonr_start")->Wert;
        $artnr_ende = (float)Parameter::model()->findByPk("artikel_autonr_ende")->Wert;

        $end_while = 0;
        while ($end_while != 1) {
            //ermittle automatisch vergebene artikelnr, eindeutigkeit mittels datenbanktransaktion sichergestellt
            $end_while = 1;
            $return = null;
            $connection = Yii::app()->db;
            //starte transaktion, jetzt kann nur diese verarbeitung auf diese tabelle zugreifen
            $transaction = $connection->beginTransaction();
            try {
                //ermittle größte artikelnummer im gültigen bereich
                $sql_max_id = "select max(artikelNr) from artikel_autonummernvergabe where artikelNr between " . $artnr_start . " and " . $artnr_ende . " and VergabeTyp='auto'";
                $max_id = $connection->createCommand($sql_max_id)->queryScalar();

                //wenn noch nicht vorhanden setze id auf startwert, sonnst + 1
                if ($max_id == null)
                    $next_id = $artnr_start;
                else
                    $next_id = $max_id + 1;

                //wenn im rahmen dann neue artikelnummer einfügen
                if ($next_id <= $artnr_ende) {
                    //neue id in db einfügen
                    $sql_insert = "insert into artikel_autonummernvergabe (ArtikelNr,VergabeTS,VergabeIP,VergabeTyp) values (" . $next_id . ",'" . date("Y-m-d H:i:s") . "','" . Verkauf::model()->getCurrentClientIP() . "','auto');";
                    $connection->createCommand($sql_insert)->execute();

                    //größte id auslesen 
                    $next_id_check = $connection->createCommand($sql_max_id)->queryScalar();

                    //prüfen ob gerade erstellt id mit wert übereinstimmt, wenn ja return
                    if ($next_id_check == $next_id) {
                        $return = $next_id;
                    }
                } else {
                    $return = null;
                }
                //beende transaktion
                $transaction->commit();
            } catch (Exception $e) {
                //beende transaktion im fehlerfall
                $transaction->rollback();
                $return = null;
            }

            //prüfe ob diese artikelnummer bereits im artikelstamm erfasst wurde, wenn ja warte 10ms und ermittle eine neue
            if ($return != null) {
                $sql_check_artikel = "select count(*) from artikel where artikelNr= " . $return . " ";
                $check_artikel = $connection->createCommand($sql_check_artikel)->queryScalar();
                if ($check_artikel == 1) {
                    $end_while = 0;
                    usleep(10000);
                }
            }
        } //while



        //id zurückgeben wenn ok, sonnst null        
        return $return;
    }

    /**
     * liefert die nächste manuell vom system vergebene artikelnummer zurück - für druck von etiketten
     */
    public function getManualArtikelNr()
    {
        //parameter einlesen (rahmenwerte für automatische artikelvergabe)
        $artnr_start = (float)Parameter::model()->findByPk("artikel_autonr_start")->Wert;
        $artnr_ende = (float)Parameter::model()->findByPk("artikel_autonr_ende")->Wert;
        $artnr_min = (float)Parameter::model()->findByPk("artikel_nr_min")->Wert;

        //sonderfall: wenn $artnr_min zwischen $artnr_start und $artnr_ende liegt wird ende angenommen
        if ($artnr_min >= $artnr_start && $artnr_min <= $artnr_ende)
            $artnr_min = $artnr_ende + 1;

        $end_while = 0;
        while ($end_while != 1) {
            //ermittle manuell vergebene artikelnr, eindeutigkeit mittels datenbanktransaktion sichergestellt
            $end_while = 1;
            $return = null;
            $connection = Yii::app()->db;
            //starte transaktion, jetzt kann nur diese verarbeitung auf diese tabelle zugreifen
            $transaction = $connection->beginTransaction();
            try {
                //ermittle größte artikelnummer im gültigen bereich
                $sql_max_id = "select max(artikelNr) from artikel_autonummernvergabe where artikelNr not between " . $artnr_start . " and " . $artnr_ende . " and VergabeTyp='man' and artikelNr >= " . $artnr_min . "";
                $max_id = $connection->createCommand($sql_max_id)->queryScalar();

                //wenn noch nicht vorhanden setze id auf startwert, sonnst + 1
                if ($max_id == null)
                    $next_id = $artnr_min; //starte mit kleinster möglicher artikelnummer aus settings
                else
                    $next_id = $max_id + 1;


                //neue id in db einfügen
                $sql_insert = "insert into artikel_autonummernvergabe (ArtikelNr,VergabeTS,VergabeIP,VergabeTyp) values (" . $next_id . ",'" . date("Y-m-d H:i:s") . "','" . Verkauf::model()->getCurrentClientIP() . "','man');";
                $connection->createCommand($sql_insert)->execute();

                //größte id auslesen 
                $next_id_check = $connection->createCommand($sql_max_id)->queryScalar();

                //prüfen ob gerade erstellt id mit wert übereinstimmt, wenn ja return
                if ($next_id_check == $next_id) {
                    $return = $next_id;
                }

                //beende transaktion
                $transaction->commit();
            } catch (Exception $e) {
                //beende transaktion im fehlerfall
                $transaction->rollback();
                $return = null;
            }

            //prüfe ob diese artikelnummer bereits im artikelstamm erfasst wurde, wenn ja warte 10ms und ermittle eine neue
            if ($return != null) {
                $sql_check_artikel = "select count(*) from artikel where artikelNr= " . $return . " ";
                $check_artikel = $connection->createCommand($sql_check_artikel)->queryScalar();
                if ($check_artikel == 1) {
                    $end_while = 0;
                    usleep(10000);
                }
            }
        } //while



        //id zurückgeben wenn ok, sonnst null        
        return $return;
    }


    /**
     * Returns the static model of the specified AR class.
     * Please note that you should have this exact method in all your CActiveRecord descendants!
     * @param string $className active record class name.
     * @return Artikel the static model class
     */
    public static function model($className = __CLASS__)
    {
        //erlaubt dann einen Zugriff mit Klasse::model()->Funktion();
        return parent::model($className);
    }


    /**
     * Importiert alle Artikel zu diesem User/Kunden aus der VorerfassungsDB
     * @param mixed $id - ID unter der der Kunde in der VorerfassungsDB aufgeführt ist
     * @param mixed $KundenNr - Der Kunde muss im System bereits mit folgender Kundennummer angelegt sein
     */
    public function importArtikelVorerfassung($id, $KundenNr)
    {
        //sicherstellen, dass id nur nummerisch ist
        $id = intval($id);

        //zugriff auf vorerfassungsDB
        $db = Helper::getVorerfassungsDB();
        $sql = "SELECT a.id as ImportID,a.name as Name,a.price as Preis,a.size as Groesse,a.brand as Marke,'Nein' as HappyHourFlag,(select b.Name from article_groups b where b.ID=a.article_group_id) as WarengruppenName,a.article_group_id as RefWarengruppeID FROM articles a where a.user_id=" . $id . "";
        $vorerfassungs_data = $db->createCommand($sql)->queryAll();

        //komische strings ausfiltern und dadurch diverse angriffe unterbinden
        $vorerfassungs_data = Helper::cleanAllInputData($vorerfassungs_data);

        //gehe durch alle artikel
        $return_ok = "";
        $return_nok = "";
        foreach ($vorerfassungs_data as $data) {
            //prüfung ob alle wichtigen daten vorhanden
            if (isset($data["ImportID"]) && isset($data["Name"]) && isset($data["Preis"]) && isset($data["HappyHourFlag"])) {
                //Prüfen ob artikel schon importiert wurde und wenn ja ersetzen und infomeldung
                $model = Artikel::model()->findByAttributes(array('ImportID' => $data["ImportID"]));
                $_POST['Artikel'] = null;

                if ($model == null) {
                    //nur bei erstimport setzen
                    $model = new Artikel;
                    $_POST['Artikel']["ImportID"] = $data["ImportID"];
                    //$_POST['Artikel']["ArtikelNr"] = $model->getAutoArtikelNr();  //artikelnummer wird automatisch vom system vergeben
                    $_POST['Artikel']["ArtikelNr"] = $data["ImportID"];  //artikelnummer wird von vorerfassung übernommen
                } else {
                    $_POST['Artikel']["ArtikelNr"] = $model->ArtikelNr; //bestehende nr setzen da unten darauf zugegriffen wird

                    //wenn bereits gesetzt info hinzufügen
                    if (Yii::app()->user->hasFlash('import_notice')) {
                        $message = Yii::app()->user->getFlash('import_notice');
                        Yii::app()->user->setFlash("import_notice", $message . "  -  ArtNr:" . $model->ArtikelNr . " (ImpID:" . $data["ImportID"] . ") war vorhanden und wurde überschrieben.");
                    } else {
                        Yii::app()->user->setFlash("import_notice", "Hinweis: ArtNr:" . $model->ArtikelNr . " (ImpID:" . $data["ImportID"] . ") war vorhanden und wurde überschrieben.");
                    }
                }
                //immer setzen
                $_POST['Artikel']["Name"] = $data["Name"];
                $_POST['Artikel']["Preis"] = $data["Preis"];
                $_POST['Artikel']["Kommentar"] = "Durch Import aus Vorerfassung erstellt";
                $_POST['Artikel']["HappyHourFlag"] = $data["HappyHourFlag"];
                $_POST['Artikel']["RefKundenNr"] = $KundenNr;
                $_POST['Artikel']["Groesse"] = $data["Groesse"];
                $_POST['Artikel']["Marke"] = $data["Marke"];

                //ermittle ID zu Warengruppenname oder lege warengruppe gleich an
                $wgr_id = Warengruppe::model()->checkWarengruppe($data["WarengruppenName"]);
                $_POST['Artikel']["RefWarengruppeID"] = $wgr_id;

                //komische strings ausfiltern und dadurch diverse angriffe unterbinden
                $_POST['Artikel'] = Helper::cleanAllInputData($_POST['Artikel']);
                //initialisieren der standardwerte etc
                $_POST['Artikel'] = $this->manipulateArtikelData($_POST['Artikel']);

                //schreibe änderungslog wenn sich mindestens ein feld verändert hat
                $model->writeChangeLog($_POST['Artikel']["ArtikelNr"]);

                $model->attributes = $_POST['Artikel'];

                //setze erfassungsts
                $model->ErfassungTS = date("Y-m-d H:i:s");


                if ($model->save()) {
                    $return_ok .= "" . $model->ArtikelNr . ", ";
                } else {
                    //$error_message=implode(array_values($model->getErrors())," ");
                    $error_array = $model->getErrors();
                    $error_message = "";
                    if ($error_array != null && is_array($error_array)) {
                        foreach ($error_array as $error_item) {
                            if (isset($error_item[0]))
                                $error_message .= $error_item[0];
                        }
                    }
                    $return_nok .= "ImportID:" . $data["ImportID"] . " - " . $error_message . ", ";
                }
            } else {
                if (isset($data["ImportID"]))
                    $return_nok .= "" . $data["ImportID"] . ", ";
                else
                    $return_nok .= "undefined, ";
            }
        } //foreach  

        if ($return_ok != "")
            $return_ok = substr($return_ok, 0, -1);
        if ($return_nok != "")
            $return_nok = " und Error bei folgenden Imports: " . substr($return_nok, 0, -1);

        return "Import Ok von ArtikelNr: " . $return_ok . $return_nok;
    }


    /**
     * Manipuliert Format von bestimmten Artikeldaten vor Speicherung/Update (Preis, Größe XL/XS etc, ... )
     * @param mixed $post - HTTP Post request zu dem Modell, zb $_:Post["Artikel"]
     */
    public function manipulateArtikelData($post)
    {
        //wenn warengruppe nicht gesetzt ist, setze default warengruppe 0 damit in changelog nicht laufend einträge
        if (isset($post["RefWarengruppeID"]) && $post["RefWarengruppeID"] == "")
            $post["RefWarengruppeID"] = "0";

        //preiseingabe wenn in komma dann mit punkt manipulieren, damit eingabe mit komma und punkt möglich
        if (isset($post["Preis"]) && $post["Preis"] != "")
            $post["Preis"] = str_replace(",", ".", $post["Preis"]);

        //wenn größe vorhanden, ersetze komma immer durch punkt
        if (isset($post["Groesse"]) && $post["Groesse"] != "")
            $post["Groesse"] = str_replace(",", ".", $post["Groesse"]);

        //größe XL, XS, etc daten sind immer uppercase
        if (isset($post["Groesse"]) && $post["Groesse"] != "" && !is_numeric($post["Groesse"]))
            $post["Groesse"] = strtoupper($post["Groesse"]);

        return $post;
    }
}
